C      *****************************************************************
C      * Subroutine SelectComp                                         *
C      * Analyzes a list of phases and determines some preliminary     *
C      * information on how they will change the basic equilibrium     *
C      * problem.  This information includes elimination of components *
C      * and two ways of determining phase order.                      *
C      *                                                               *
C      * Variables:                                                    *
C      * AvailComp -Local - Indicates whether component is available   *
C      *                    (TRUE) or not (FALSE).  Unitless.          *
C      * C         -Local - Array index.  Unitless.                    *
C      * CompThere -Local - Boolean flag.  Unitless.                   *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * Index     -Local - A vector, of size NPhases, that holds the  *
C      *                    sorting order.  Unitless.                  *
C      * LargeNum  -Local - A large number.  Unitless.                 *
C      * MasBal    -Input - Indicates a species that is limited by     *
C      *                    mass balance.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinC      -Local - Temporary storage.  Unitless.              *
C      * MinS      -Local - Temporary storage.Unitless.                *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NCompSize -Input - The max number of components, used to size *
C      *                    arrays.  Unitless.                         *
C      *                    (file ArraySizes.Inc)                      *
C      * NPhases   -Input - The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NumComp   -Local - An vector, of size NPhases, that indicates *
C      *                    indicates how many components are available*
C      *                    for each phase.  Unitless.                 *
C      * NumDone   -Local - Boolean flag.  Unitless.                   *
C      * One       -Local - The number one.  Unitless.                 *
C      * Order     -Local - Array index.  Unitless.                    *
C      * P         -Local - Array index.  Unitless.                    *
C      * PhaseAvail-Local - This array indicates whether the phase is  *
C      *                    available or not.  Unitless.               *
C      * PickNext  -Fcn   - Integer value of index of next available   *
C      *                    item.  Unitless.                           *
C      * POrder    -Output- A vector, of size NPhases, of order in     *
C      *                    which to consider phases in mole balance   *
C      *                    calculations.  Unitless.                   *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * Q         -Local - Array index.  Unitless.                    *
C      * RefP      -Local - Array index.  Unitless.                    *
C      * SComp     -Output- A vector, of size NPhases, of the          *
C      *                    component associated with each phase.      *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SolGas    -Input - Indicates a species that is solid or       *
C      *                    gaseous controlled.  Unitless.             *
C      *                    (Common block VParam, file VModel.f)       *
C      * SOrder    -Output- A vector, of size NPhases, of order in     *
C      *                    which to consider phases in mass action    *
C      *                    substitutions.  Unitless.                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SP        -Input - A matrix, of size NPhases by NComps, of    *
C      *                    coefficients.  Unitless.                   *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SPhase    -Output- A vector, of size NComp, of phase          *
C      *                    associated with each component.  Unitless. *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SolGas    -Input - Indicates a species that is solid or       *
C      *                    gaseous controlled.                        *
C      *                    (common block VParam, file VModel.f)       *
C      * SPMasBal  -Input - Indicates a solid phase species type       *
C      *                    limited by mass balance.  Unitless.        *
C      *                    (Common block VModel, file VModel.f)       *
C      * Start     -Local - Array index.  Unitless.                    *
C      * SType     -Input - A vector, of size NSpecies, of each        *
C      *                    species's type.  Unitless.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      * TestC     -Local - Component index for testing associations   *
C      *                    with phases.  Unitless.                    *
C      * TestP     -Local - Phase index for testing associations       *
C      *                    with components.  Unitless.                *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE SELECTCOMP(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'

       LOGICAL AVAILCOMP(NCOMPSIZE), COMPTHERE, PHASEAVAIL(NPHASESSIZE)

       INTEGER C, GOAHEAD, INDEX(NPHASESSIZE)
       INTEGER MINC, N, NUMCOMP(NPHASESSIZE), NUMDONE, ONE
       INTEGER ORDER, P, PICKNEXT, Q, REFP, START, TESTC, TESTP

       REAL*8  LARGE_NUM, MINS

       PARAMETER (LARGE_NUM = 10000, ONE = 1)

C      *------------------------------------------*
C      * Indicate that no errors have been found. *
C      *------------------------------------------*
       GOAHEAD = 1

C      *----------------------------------------------------------*
C      * Components that are not fixed concentration are noted as *
C      * available.                                               *
C      *----------------------------------------------------------*
       DO 100 C=1, NCOMP
          IF ((STYPE(C).EQ.MASBAL).OR.(STYPE(C).EQ.SPMASBAL)) THEN
             AVAILCOMP(C) = .TRUE.
          ELSE
             AVAILCOMP(C) = .FALSE.
          ENDIF
  100  CONTINUE

C      *-------------------------------------*
C      * Initialize index array for sorting. *
C      *-------------------------------------*
       DO 110 P=1, NPHASES
          INDEX(P) = P
  110  CONTINUE

C      *---------------------------------------------------------------*
C      * Fill in array which stores the number of components available *
C      * for each phase.                                               *
C      *---------------------------------------------------------------*
       START = 0

C      *-----------------------------------------------------------*
C      * The following line starts a DO loop that ends on line 160 *
C      *-----------------------------------------------------------*
  120  CONTINUE
          START = START + 1
C         *----------------------------------------------------------*
C         * For each phase, count the number of components available *
C         * for substitution.                                        *
C         *----------------------------------------------------------*
          DO 140 Q = 1, NPHASES
             P = INDEX(Q)
             N = 0
             DO 130 C=1, NCOMP
                IF ((SP(P,C).NE.0).AND.(AVAILCOMP(C))) N = N + 1
  130        CONTINUE
C            *----------------------------------------------------*
C            * If N is zero this is a terminal error, Gibbs phase *
C            * rule violation in list of phases.  An error message*
C            * is output and a flag is passed back to caller which*
C            * should result in program termination.              *
C            *----------------------------------------------------*
             IF (N.EQ.0) THEN
                WRITE(6,*) 'Fatal error in routine SelectComp'
                WRITE(6,*) 'Gibbs phase rule violation.'
                WRITE(6,*) 'Program terminates.'
                WRITE(YASOUT,*) 'Fatal error in routine SelectComp'
                WRITE(YASOUT,*) 'Gibbs phase rule violation.'
                WRITE(YASOUT,*) 'Program terminates.'
                GOAHEAD = 0
                GOTO 9999
             ENDIF
             NUMCOMP(P) = N
  140     CONTINUE

C         *----------------------------------------------------------*
C         * Sort the list of phases in ascending number of available *
C         * components and store the sorted lists in the list INDEX. *
C         *----------------------------------------------------------*
          CALL INDSORT(NUMCOMP, INDEX, NPHASES)

C         *-------------------------------------------------*
C         * Get phase with the smallest number of available *
C         * components.                                     *
C         *-------------------------------------------------*
          P = INDEX(START)

C         *-----------------------------------------------------------*
C         * Find the component with the smallest non-zero coefficient.*
C         *-----------------------------------------------------------*
          MINC = 0
          MINS = LARGE_NUM
          DO 150 C = 1, NCOMP
             IF ((AVAILCOMP(C)).AND.(ABS(SP(P,C)).LE.MINS).AND.
     >       (SP(P,C).NE.0.0)) THEN
                MINS = ABS(SP(P,C))
                MINC = C
             ENDIF
  150     CONTINUE

C         *---------------------------------------------------------*
C         * This is the component to be associated with this phase. *
C         * Store this component number in the SPhase list.  Store  *
C         * this phase number in the SComp list.                    *
C         *---------------------------------------------------------*
          IF (MINC.NE.0) THEN
             SPHASE(MINC) = INDEX(START)
             SCOMP(INDEX(START)) = MINC
             AVAILCOMP(MINC) = .FALSE.
C            *----------------------------------------------------*
C            * Change the SType of this component if it currently *
C            * is a mass balance SType.                           *
C            *----------------------------------------------------*
             IF (STYPE(MINC).EQ.MASBAL) THEN 
                STYPE(MINC) = SOLGAS
             ELSEIF (STYPE(MINC).EQ.SPMASBAL) THEN
                 STYPE(MINC) = SPSOLGAS
             END IF
          ELSE
C            *-----------------------------------------------------*
C            * This should never happen.  If it does program quits *
C            * here with error message.                            *
C            *-----------------------------------------------------*
             WRITE(6,*) 'In routine SelectComp'
             WRITE(6,*) 'a fatal error has occured, there is no'
             WRITE(6,*) 'smallest non-zero coefficient.'
             WRITE(6,*) 'Investigate immediately!'
             WRITE(YASOUT,*) 'In routine SelectComp'
             WRITE(YASOUT,*) 'a fatal error has occured, there is no'
             WRITE(YASOUT,*) 'smallest non-zero coefficient.'
             WRITE(YASOUT,*) 'Investigate immediately!'
             GOAHEAD = 0
             STOP
         ENDIF

C        *-----------------------------------------------------------*
C        * The following line is the bottom of the DO loop beginning *
C        * with line 120.                                            *
C        *-----------------------------------------------------------*
  160  CONTINUE
       IF (START.NE.NPHASES) GOTO 120

C      *--------------------------------------------------------------*
C      * Order the list of phases two ways: First, determine an order *
C      * for interdependent phase transformations for mole balance    *
C      * calculations (POrder).  Strategy:                            *
C      * 1) Select a phase from the list to consider, note its        *
C      *    associated component.                                     *
C      * 2) Examine remaining list of phases to see if there is a non-*
C      *    zero coefficient (variable SP()) for that component.      *
C      * 3) If no other such phase is found, then add this phase to   *
C      *    the list POrder and remove it from the list of phases to  *
C      *    consider.                                                 *
C      * 4) If another phase is found, then ignore this phase for now *
C      * 5) If the list of phases to consider is not empty then go    *
C      *    back to step (1).                                         *
C      *--------------------------------------------------------------*
       REFP = 0
       ORDER = 1
       NUMDONE = 0

C      *----------------------------------*       
C      * Clear the phase available array. *
C      *----------------------------------*
       DO 165 P = 1, NPHASES
          PHASEAVAIL(P) = .TRUE.
  165  CONTINUE

C      *------------------------------------------------------------*
C      * The following line starts a DO loop that ends on line 200. *
C      *------------------------------------------------------------*
  170  CONTINUE
          REFP = PICKNEXT(ONE, NPHASES, PHASEAVAIL, REFP, GOAHEAD)
          IF (GOAHEAD.EQ.0) THEN
             WRITE(6,*) 'Fatal error in routine SelectComp'
             WRITE(6,*) 'on return from PickNext.'
             WRITE(YASOUT,*) 'Fatal error in routine SelectComp'
             WRITE(YASOUT,*) 'on return from PickNext.'
             GOTO 9999
          ENDIF

          TESTC = SCOMP(REFP)
          COMPTHERE = .FALSE.
          TESTP = 1

C         *--------------------------------------------------------*
C         * The next line starts a DO loop that ends on line 190.  *
C         *--------------------------------------------------------*
  180     CONTINUE
             IF ((PHASEAVAIL(TESTP)).AND.
     >       (SP(TESTP,TESTC).NE.0).AND.(TESTP.NE.REFP)) THEN
                COMPTHERE = .TRUE.
             ENDIF
             TESTP = TESTP + 1

C         *----------------------------------------------------*
C         * The following line ends DO loop begining line 180. *
C         *----------------------------------------------------*
  190     IF (TESTP.LE.NPHASES) GOTO 180

C         *----------------------------------------------------*
C         * If the criterion is met move phase to POrder list. *
C         *----------------------------------------------------*
          IF (.NOT.COMPTHERE) THEN
             PHASEAVAIL(REFP) = .FALSE.
             PORDER(ORDER) = REFP
             ORDER = ORDER + 1
             NUMDONE = NUMDONE + 1
          ENDIF

C      *----------------------------------------------------*
C      * The next line is the bottom of a DO loop beginning *
C      * line 170.                                          *
C      *----------------------------------------------------*
  200  IF (NUMDONE.LT.NPHASES) GOTO 170

C      *--------------------------------------------------------*
C      * Now determine the order for substituting components    *
C      * associated with phases in mass action expressions.     *
C      * Strategy:                                              *
C      * 1) Take a phase from the list of available phases.     *
C      * 2) Find every component (except SComp) with a non-zero *
C      *    coefficient.                                        *
C      * 3) For every component found, determine if it is       *
C      *    associated (via SComp) with another phase.          *
C      * 4) If no component is associated with another phase    *
C      *    then move this phase to the list SOrder and remove  *
C      *    it from available list.                             *
C      * 5) If available list is not empty begin again at step  *
C      *    one.                                                *
C      *--------------------------------------------------------*

C      *----------------------------------*       
C      * Clear the phase available array. *
C      *----------------------------------*
       DO 205 P = 1, NPHASES
          PHASEAVAIL(P) = .TRUE.
  205  CONTINUE

       REFP = 0
       ORDER = 1
       NUMDONE = 0

C      *-------------------------------------------------------*       
C      * The next line begins a DO loop that ends on line 230. *
C      *-------------------------------------------------------* 
  210  CONTINUE
          REFP = PICKNEXT(ONE, NPHASES, PHASEAVAIL, REFP, GOAHEAD)

C         *------------------------------*
C         * Check all components in RefP *
C         *------------------------------*
          COMPTHERE = .FALSE.
          DO 220 C = 1, NCOMP
             IF ((SP(REFP,C).NE.0).AND.
     >           (C.NE.SCOMP(REFP)).AND.
     >           (SPHASE(C).NE.0)) THEN
                IF (PHASEAVAIL(SPHASE(C))) THEN
                   COMPTHERE = .TRUE.
                ENDIF
              ENDIF
  220     CONTINUE

          IF (.NOT.COMPTHERE) THEN
             PHASEAVAIL(REFP) = .FALSE.
             SORDER(ORDER) = REFP
             ORDER = ORDER + 1
             NUMDONE = NUMDONE + 1
          ENDIF

C      *-------------------------------------------------------*
C      * The following line is the bottom of DO loop beginning *
C      * on line 210.                                          *
C      *-------------------------------------------------------*
       IF (NUMDONE.LT.NPHASES) GOTO 210

C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE
             
	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
